// Supports UVM-1.1d, needs your update to support UVM-1.2/UVM-IEEE
// Look for "Update here"
package cvm_abort_package;
    import uvm_pkg::*;
    `include "uvm_macros.svh"

    class cvm_phase_monitor extends uvm_component;
        `uvm_component_utils(cvm_phase_monitor)
        local static cvm_phase_monitor m;
        function new(string name = "cvm_phase_monitor", 
                     uvm_component parent = null);
            super.new(name, parent);
        endfunction
        static function cvm_phase_monitor get();
          if(m == null)
              m = cvm_phase_monitor::type_id::create("mon", uvm_top);
          return m;
        endfunction

        typedef uvm_phase phase_map_t[uvm_phase]; // imp |-> node
        typedef phase_map_t domain_map_t[uvm_domain];
        static domain_map_t m_current_phase[cvm_phase_monitor];

        virtual function void phase_started(uvm_phase phase);
            uvm_domain dom = phase.get_domain();
            uvm_phase imp = phase.get_imp();
            if(!m_current_phase.exists(this)) begin
                domain_map_t mon2domain; // new monitor
                m_current_phase[this] = mon2domain;
            end
            if(!m_current_phase[this].exists(dom)) begin
                phase_map_t domain2phase; // new domain
                m_current_phase[this][dom] = domain2phase;
            end
            if(!m_current_phase[this][dom].exists(imp)) begin
                m_current_phase[this][dom][imp] = phase; // rec'd
            end
        endfunction

        virtual function void phase_ended(uvm_phase phase);
            uvm_domain dom = phase.get_domain();
            uvm_phase imp = phase.get_imp();
            if(!m_current_phase.exists(this)) begin end
            else if(!m_current_phase[this].exists(dom)) begin end
            else if(!m_current_phase[this][dom].exists(imp)) begin end
            else begin
                m_current_phase[this][dom].delete(imp); // delete phase
                if(m_current_phase[this][dom].size() == 0) begin
                    m_current_phase[this].delete(dom); // delete empty domain
                    if(m_current_phase[this].size() == 0) 
                        m_current_phase.delete(this); // delete empty array
                end
            end
        endfunction

        static function uvm_phase curr_phase();
            uvm_task_phase is_task; uvm_phase result = null;
            if(m_current_phase.exists(m)) begin
                domain_map_t curr_d = m_current_phase[m];
                foreach(curr_d[dom]) begin
                    phase_map_t curr_ph = curr_d[dom];
                    foreach(curr_ph[imp]) begin
                        if($cast(is_task, imp)) // prefer task phase
                            return curr_ph[imp];
                        else if(result == null)
                            result = curr_ph[imp]; // otherwise take first phase
                    end
                end
            end
            return result;
        endfunction
    endclass


    class cvm_abort_handler extends uvm_report_object;
        `uvm_object_utils(cvm_abort_handler)
        local static cvm_abort_handler m;
        protected cvm_phase_monitor ph;

        function new(string name = "cvm_abort_handler");
            super.new(name);
            ph = cvm_phase_monitor::get();
        endfunction

        static function cvm_abort_handler get();
            if(m == null) 
                m = cvm_abort_handler::type_id::create("abt", null);
            return m;
        endfunction

        static bit m_external_die = 0;
        static int unsigned m_val = 1;
        static string m_unit = "ns";
        // maybe implement a function to manipulate the value and unit?

        static task delay_finish();
            case(m_unit)
                "ms": #(m_val * 1ms) $finish;
                "us": #(m_val * 1us) $finish;
                "ps": #(m_val * 1ps) $finish;
                "fs": #(m_val * 1fs) $finish;
                default: #(m_val * 1ns) $finish;
            endcase
        endtask

        virtual function void die();
            uvm_root root = uvm_root::get();
            uvm_phase phase = ph.curr_phase();
            uvm_task_phase is_task;
            root.m_do_pre_abort();
            report_summarize();
            if((phase != null) && ($cast(is_task, phase.get_imp()))) begin
                phase.raise_objection(root); // prevent phase exit
                fork delay_finish(); join_none
            end else begin
                root.finish_on_completion = 0;
                m_external_die = 1;
            end
        endfunction

        static task die_external();
            if(m_external_die) delay_finish();
        endtask
    endclass

    class cvm_report_server 
`ifdef UVM_VERSION_1_1
    extends uvm_report_server;
`elsif UVM_VERSION_1_2
    extends uvm_default_report_server;
`endif
        local static cvm_report_server m;
        static cvm_abort_handler abt;
        function new();
            abt = cvm_abort_handler::get();
        endfunction
        static function cvm_report_server get();
            if(m == null) m = new();
            return m;
        endfunction
        static function uvm_report_server get_server();
            return get();
        endfunction
        static function void install_server();
            uvm_report_server cvm_srvr = cvm_report_server::get_server();
            uvm_report_server uvm_srvr = uvm_report_server::get_server();
            if(cvm_srvr != uvm_srvr) begin
                uvm_report_server::set_server(cvm_srvr);
                `uvm_info("CVM_RPT_SRVR",
                          "Custom report server installed", UVM_NONE)
            end
        endfunction

`ifdef UVM_VERSION_1_1
        virtual function void process_report(uvm_severity severity, string name, 
                                 string id, string message, uvm_action action, 
                                 UVM_FILE file, string filename, int line,
                                 string composed_message, int verbosity_level, 
                                 uvm_report_object client);

            if((action & UVM_EXIT) && (abt != null))
                client = abt;
            super.process_report(severity, name, id, message, action, file,
                                 filename, line, composed_message, 
                                 verbosity_level, client);
        endfunction
`elsif UVM_VERSION_1_2
        // Update here to support UVM-1.2
        //   1. Copy the following function from your UVM-1.2 library to here
        //   2. Modify the exit action to the following
        //virtual function void execute_report_message(
        //                         uvm_report_message report_message,
        //                         string composed_message);
        //
        // ...copy-and-paste...
        //
        //  // Process the UVM_EXIT action
        //  if(report_message.get_action() & UVM_EXIT) begin
        //    if(abt == null) begin // insert this
        //     uvm_root l_root;
        //     uvm_coreservice_t cs;
        //     cs = uvm_coreservice_t::get();
        //     l_root = cs.get_root();
        //     l_root.die();
        //    end else abt.die();  // and insert this
        //  end
        //
        // ...copy-and-paste...
        //
        //endfunction
`endif

    endclass

endpackage


